﻿using System;
using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//CollectionInfoMenu handles the collection info UI
public class CollectionInfoMenu : MonoBehaviour
{
    //UI Elements
    public GameObject CollectionInfoUI;
    public TMP_Text DescriptionText;
    public TMP_Dropdown DifficultiesDropdown;
    public TMP_Text TimeText;
    public TMP_Text MoneyText;
    public TMP_Text ItemsText;
    public Animator Animator;

    private CollectionMission _Mission;

    /// <summary>
    /// Shows the UI for the corresponding collection mission
    /// </summary>
    /// <param name="mission">The mission to show info for</param>
    public void Show(CollectionMission mission)
    {
        _Mission = mission; //Store the mission locally

        //Setup the UI, begin the audio and animation
        SetupDropdown();
        OnDifficultyChanged(0);
        CollectionInfoUI.SetActive(true);
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransToAudioID]);
        Animator.Play("Show");
    }

    /// <summary>
    /// Hides the UI
    /// </summary>
    public void Hide()
    {
        //Play the UI SFX, begin the animation
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransFromAudioID]);
        Animator.Play("Hide");
    }

    /// <summary>
    /// Handler for when the hide animation completes
    /// </summary>
    public void OnHideAnimationCompleted()
    {
        //All done, hide our UI and the base info menu as well
        CollectionInfoUI.SetActive(false);
        GameManager.Instance.MissionInfoMenu.Hide();
    }

    /// <summary>
    /// Handler for when the selected difficulty changes
    /// </summary>
    /// <param name="index">Thhe index of the new difficulty selected</param>
    public void OnDifficultyChanged(int index)
    {
        Constants.MissionDifficulties difficulty = GetSelectedDifficulty();

        if (difficulty != Constants.MissionDifficulties.Sentinel)
        {
            //The selected value is valid, set the UI
            MoneyText.text = "$" + _Mission.DifficultyData[difficulty].Money;

            TimeSpan time = TimeSpan.FromMilliseconds(_Mission.DifficultyData[difficulty].Time);
            TimeText.text = string.Format("{0}:{1:00}", (int)time.TotalMinutes, time.Seconds);

            ItemsText.text = _Mission.DifficultyData[difficulty].ItemPositions.Count.ToString();
        }
    }

    /// <summary>
    /// Handler for the accept button
    /// </summary>
    public void OnMissionAccepted()
    {
        Constants.MissionDifficulties selectedDifficulty = GetSelectedDifficulty();

        if (selectedDifficulty != Constants.MissionDifficulties.Sentinel)
        {
            //The difficulty is valid, let's start
            MissionsManager.Instance.StartMission(_Mission, selectedDifficulty);
        }

        else
        {
            Debug.LogError("ERROR: Failed to start mission as the selected difficulty was sentinel.");
        }

        Hide(); //Time to hide
    }

    /// <summary>
    /// Handler for the cancel button
    /// </summary>
    public void OnMissionCancelled()
    {
        Hide(); //Time to hide
    }

    /// <summary>
    /// Sets up the dropdown by looping through all the enum values and converting them to be displayed
    /// </summary>
    private void SetupDropdown()
    {
        DifficultiesDropdown.ClearOptions();

        foreach (Constants.MissionDifficulties difficulties in Enum.GetValues(typeof(Constants.MissionDifficulties)))
        {
            if (difficulties != Constants.MissionDifficulties.Sentinel)
            {
                DifficultiesDropdown.options.Add(new TMP_Dropdown.OptionData() { text = difficulties.ToString() });
            }
        }

        DifficultiesDropdown.captionText.text = DifficultiesDropdown.options[0].text;
    }

    /// <summary>
    /// Gets the selected difficulty as an enum value by parsing the string
    /// </summary>
    /// <returns>The enum difficulty value if successful, otherwise sentinel</returns>
    private Constants.MissionDifficulties GetSelectedDifficulty()
    {
        Constants.MissionDifficulties retDifficulty = Constants.MissionDifficulties.Sentinel;
        string selectedOptionText = DifficultiesDropdown.options[DifficultiesDropdown.value].text;

        if (!Enum.TryParse(selectedOptionText, out retDifficulty))
        {
            return Constants.MissionDifficulties.Sentinel;
        }

        else
        {
            return retDifficulty;
        }
    }

    private void Update()
    {
        if (Input.GetKeyDown(KeyCode.Escape))
        {
            Hide();
        }
    }
}
